<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

// Check if user is logged in
requireLogin();

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

try {
    if (!isset($_POST['id']) || empty($_POST['id'])) {
        throw new Exception('Quotation ID is required');
    }
    
    $quotation_id = intval($_POST['id']);
    
    // Check if quotation exists and get details
    $stmt = $pdo->prepare("
        SELECT quote_number, status, created_by 
        FROM quotations 
        WHERE id = ?
    ");
    $stmt->execute([$quotation_id]);
    $quotation = $stmt->fetch();
    
    if (!$quotation) {
        throw new Exception('Quotation not found');
    }
    
    // Check if user can delete this quotation
    // Only super admins or the creator can delete
    $stmt = $pdo->prepare("SELECT is_super_admin FROM roles WHERE id = (SELECT role_id FROM users WHERE id = ?)");
    $stmt->execute([$_SESSION['user_id']]);
    $isSuperAdmin = $stmt->fetchColumn();
    
    if (!$isSuperAdmin && $quotation['created_by'] != $_SESSION['user_id']) {
        throw new Exception('You do not have permission to delete this quotation');
    }
    
    // Check if quotation has been converted to sale
    if ($quotation['status'] === 'converted') {
        throw new Exception('Cannot delete a quotation that has been converted to a sale');
    }
    
    // Start transaction
    $pdo->beginTransaction();
    
    try {
        // Delete quotation items first (foreign key constraint)
        $stmt = $pdo->prepare("DELETE FROM quotation_items WHERE quotation_id = ?");
        $stmt->execute([$quotation_id]);
        
        // Delete quotation
        $stmt = $pdo->prepare("DELETE FROM quotations WHERE id = ?");
        $stmt->execute([$quotation_id]);
        
        // Commit transaction
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Quotation ' . $quotation['quote_number'] . ' deleted successfully'
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?> 